(* large integers, solyga@gmx.de, 2025-07-29 .. 2025-12-16 *)
IMPLEMENTATION MODULE Ints;

IMPORT Types, Str, Lib2, LC;

(*** operations ***)

PROCEDURE Add( x-, y-: TValue; VAR z: TValue ): BOOLEAN;
BEGIN
  IF x.n = y.n THEN
    z.n := x.n;
    IF ~ LC.Add( x.v, y.v, z.v ) THEN RETURN FALSE END;
  ELSIF LC.Cmp( x.v, y.v ) >= 0 THEN
    z.n := x.n;
    IF ~ LC.Sub( x.v, y.v, z.v ) THEN RETURN FALSE END;
  ELSE
    z.n := y.n;
    IF ~ LC.Sub( y.v, x.v, z.v ) THEN RETURN FALSE END;
  END;
  IF LC.Len0( z.v ) = 0 THEN z.n := FALSE END;  (* -0 -> +0 *)
  RETURN TRUE;
END Add;

PROCEDURE Sub( x-, y: TValue; VAR z: TValue ): BOOLEAN;
BEGIN
  Neg( y );
  RETURN Add( x, y, z );
END Sub;

PROCEDURE Mul( x-, y-: TValue; VAR z: TValue ): BOOLEAN;
BEGIN
  z.n := x.n # y.n;
  RETURN LC.Mul( x.v, y.v, z.v );
END Mul;

(* truncated = rounding towards zero *)
PROCEDURE DivT( x-, y-: TValue; VAR z, r: TValue ): BOOLEAN;
BEGIN
  z.n := x.n # y.n;
  r.n := x.n;
  RETURN LC.Div( x.v, y.v, z.v, r.v );
END DivT;

(* floored = rounding towards minus infinity *)
PROCEDURE DivF( x-, y-: TValue; VAR z, r: TValue ): BOOLEAN;
BEGIN
  z.n := x.n # y.n;
  r.n := y.n;
  IF ~ LC.Div( x.v, y.v, z.v, r.v ) THEN RETURN FALSE END;
  IF z.n & ( LC.Len0( r.v ) > 0 ) THEN
    IF ~ LC.Inc( z.v, TSht( 1 ) ) OR ~ LC.Sub( y.v, r.v, r.v ) THEN RETURN FALSE END;
  END;
  RETURN TRUE;
END DivF;

(* euclidean = non-negative remainder *)
PROCEDURE DivE( x-, y-: TValue; VAR z, r: TValue ): BOOLEAN;
BEGIN
  z.n := x.n # y.n;
  r.n := FALSE;
  IF ~ LC.Div( x.v, y.v, z.v, r.v ) THEN RETURN FALSE END;
  IF x.n & ( LC.Len0( r.v ) > 0 ) THEN
    IF ~ LC.Inc( z.v, TSht( 1 ) ) OR ~ LC.Sub( y.v, r.v, r.v ) THEN RETURN FALSE END;
  END;
  RETURN TRUE;
END DivE;

(* cleiling = rounding towards plus infinity *)
PROCEDURE DivC( x-, y-: TValue; VAR z, r: TValue ): BOOLEAN;
BEGIN
  z.n := x.n # y.n;
  r.n := ~ y.n;
  IF ~ LC.Div( x.v, y.v, z.v, r.v ) THEN RETURN FALSE END;
  IF ~ z.n & ( LC.Len0( r.v ) > 0 ) THEN
    IF ~ LC.Inc( z.v, TSht( 1 ) ) OR ~ LC.Sub( y.v, r.v, r.v ) THEN RETURN FALSE END;
  END;
  RETURN TRUE;
END DivC;

(* rounding to nearest, half to even *)
PROCEDURE DivR( x, y-: TValue; VAR z, r: TValue ): BOOLEAN;
VAR
  c: INTEGER;
BEGIN
  z.n := x.n # y.n;
  r.n := x.n;
  IF ~ LC.Div( x.v, y.v, z.v, r.v ) OR ~ LC.Sub( y.v, r.v, x.v ) THEN RETURN FALSE END;
  c := LC.Cmp( r.v, x.v );
  IF ( c > 0 ) OR ( c = 0 ) & LC.Odd( z.v ) THEN
    IF ~ LC.Inc( z.v, TSht( 1 ) ) OR ~ LC.Cpy( r.v, x.v ) THEN RETURN FALSE END;
    r.n := ~ r.n;
  END;
  RETURN TRUE;
END DivR;


PROCEDURE Zero( VAR x: TValue );
BEGIN
  Lib2.Fill( x, 0 );
END Zero;

PROCEDURE Neg( VAR x: TValue );
BEGIN
  x.n := ~ x.n;
END Neg;

PROCEDURE Inc( VAR x: TValue; y-: TValue ): BOOLEAN;
BEGIN
  RETURN Add( x, y, x );
END Inc;

PROCEDURE Dec( VAR x: TValue; y-: TValue ): BOOLEAN;
BEGIN
  RETURN Sub( x, y, x );
END Dec;

PROCEDURE MulC( x, y-: TValue; VAR z: TValue ): BOOLEAN;
BEGIN
  RETURN Mul( x, y, z );
END MulC;

PROCEDURE Mpl( VAR x: TValue; y-: TValue ): BOOLEAN;
BEGIN
  RETURN MulC( x, y, x );
END Mpl;

PROCEDURE DvdT( VAR x: TValue; y-: TValue; VAR r: TValue ): BOOLEAN;
BEGIN
  RETURN DivT( x, y, x, r );
END DvdT;

PROCEDURE DvdF( VAR x: TValue; y-: TValue; VAR r: TValue ): BOOLEAN;
BEGIN
  RETURN DivF( x, y, x, r );
END DvdF;

PROCEDURE DvdE( VAR x: TValue; y-: TValue; VAR r: TValue ): BOOLEAN;
BEGIN
  RETURN DivE( x, y, x, r );
END DvdE;

PROCEDURE DvdC( VAR x: TValue; y-: TValue; VAR r: TValue ): BOOLEAN;
BEGIN
  RETURN DivC( x, y, x, r );
END DvdC;

PROCEDURE DvdR( VAR x: TValue; y-: TValue; VAR r: TValue ): BOOLEAN;
BEGIN
  RETURN DivR( x, y, x, r );
END DvdR;

(*** tests ***)

PROCEDURE IsZero( x: TValue ): BOOLEAN;
BEGIN
  RETURN LC.Len0( x.v ) = 0;
END IsZero;

(*** conversion ***)

PROCEDURE Put( VAR x: TValue; d, b: TSht ): BOOLEAN;
BEGIN
  RETURN LC.Mpl( x.v, b ) & LC.Inc( x.v, d );
END Put;

PROCEDURE Get( VAR x: TValue; b: TSht; VAR r: TSht ): BOOLEAN;
BEGIN
  RETURN LC.Dvd( x.v, b, r );
END Get;


PROCEDURE Digit( c: CHAR ): TSht;
BEGIN
  c := CAP( c );
  CASE c OF
  | '0'..'9': RETURN ORD( c ) - ORD( '0' );
  | 'A'..'Z': RETURN ORD( c ) - ORD( 'A' ) + 10;
  ELSE
    RETURN bMax;
  END;
END Digit;

PROCEDURE MkVal( VAR x: TValue; s-: ARRAY OF CHAR; b: TSht ): BOOLEAN;
VAR
  i: CARDINAL;
  d: TSht;
BEGIN
  IF ( b < bMin ) OR ( b > bMax ) THEN RETURN FALSE END;
  IF ( s[0] = '-' ) OR ( s[0] = '+' ) THEN i := 1 ELSE i := 0 END;
  IF s[i] = 0C THEN RETURN FALSE END;;
  Zero( x );
  WHILE ( i < SIZE(s) ) & ( s[i] # 0C ) DO
    d := Digit( s[i] );
    IF ( d >= b ) OR ~ Put( x, d, b ) THEN RETURN FALSE END;
    INC( i );
  END;
  IF s[0] = '-' THEN Neg( x ) END;
  RETURN TRUE;
END MkVal;


PROCEDURE Char( x: TSht ): CHAR;
BEGIN
  IF x < 10 THEN
    RETURN CHR( x + ORD('0') );
  ELSE
    RETURN CHR( x - 10 + ORD('A') );
  END;
END Char;

PROCEDURE Invert( VAR s: ARRAY OF CHAR; l: CARDINAL );
VAR
  i, j: CARDINAL;
  c: CHAR;
BEGIN
  IF l > SIZE( s ) THEN l := SIZE( s ) END;
  IF l < 2 THEN RETURN END;
  FOR i := 0 TO l DIV 2 - 1 DO
    j := l - 1 - i;
    c := s[ i ]; s[ i ] := s[ j ]; s[ j ] := c;
  END;
END Invert;

PROCEDURE MkStr( VAR s: ARRAY OF CHAR; x: TValue; b: TSht ): BOOLEAN;
VAR
  l: CARDINAL;

  PROCEDURE Put( c: CHAR ): BOOLEAN;
  BEGIN
    IF l >= SIZE( s ) THEN RETURN FALSE END;
    s[ l ] := c; INC( l );
    RETURN TRUE;
  END Put;

VAR
  r: TSht;
BEGIN
  IF ( b < bMin ) OR ( b > bMax ) THEN RETURN FALSE END;
  l := 0;
  REPEAT
    IF ~ Get( x, b, r ) OR ~ Put( Char(r) ) THEN RETURN FALSE END;
  UNTIL IsZero( x );
  IF x.n & ~  Put( '-' ) THEN RETURN FALSE END;
  IF l < SIZE( s ) THEN s[ l ] := 0C END;
  Invert( s, l );
  RETURN TRUE;
END MkStr;

END Ints.
